const { app, BrowserWindow, dialog, ipcMain, shell, session } = require('electron');
const path = require('path');
const fs = require('fs');
const fsPromises = fs.promises;
const chokidar = require('chokidar');
const crypto = require('crypto');
const os = require('os');

const logger = require('./src/main/logger');
const { resolveIconPath, updateWindowsShortcutIcons } = require('./src/main/icon');
const settingsStore = require('./src/main/settings');
const videoUtils = require('./src/main/videos');
const { createMenu } = require('./src/main/menu');
const { createDirectoryWatcher } = require('./src/main/watchers');
const ffmpeg = require('./src/main/ffmpeg');
const { registerIpcHandlers } = require('./src/main/ipc');
const googleOAuth = require('./src/main/auth/googleOAuth');
const YouTubeService = require('./src/main/youtube');
const analytics = require('./src/main/analytics');

let transcoderModule = null;
let stopTranscoderService = null;
let lastDirectorySummary = {
  directory_count: 0,
  network_path_count: 0,
  has_custom_labels: 0
};

try {
  transcoderModule = require('./server');
} catch (error) {
  logger.warn('Live transcoder module unavailable:', error && error.message ? error.message : error);
}

const iconPath = resolveIconPath();
const APP_USER_MODEL_ID = 'com.miba.video.manager';

if (process.platform === 'win32') {
  app.setAppUserModelId(APP_USER_MODEL_ID);
}

let mainWindow = null;
let unsubscribeRegistryListener = null;

const directoryWatcher = createDirectoryWatcher({
  chokidar,
  videoExtensions: videoUtils.videoExtensions,
  scanDirectory: videoUtils.scanDirectory,
  getMainWindow: () => mainWindow
});

const youtubeService = new YouTubeService({
  app,
  googleOAuth,
  getMainWindow: () => mainWindow
});

(async function bootstrapMainProcess() {
  try {
    await analytics.bootstrapEventRegistry();
  } catch (error) {
    logger.warn(
      'Failed to bootstrap analytics event registry:',
      error && error.message ? error.message : error
    );
  }

  analytics.captureMainEvent('application_start', {
    app_version: typeof app.getVersion === 'function' ? app.getVersion() : null,
    platform: os.platform(),
    platform_release: os.release()
  });

  registerIpcHandlers({
    ipcMain,
    dialog,
    fs,
    fsPromises,
    path,
    crypto,
    app,
    videoUtils,
    settingsStore,
    directoryWatcher,
    ffmpeg,
    getMainWindow: () => mainWindow,
    shell,
    googleOAuth,
    youtubeService
  });

  analytics.captureMainEvent('application_start', {
    app_version: typeof app.getVersion === 'function' ? app.getVersion() : null,
    platform: os.platform(),
    platform_release: os.release()
  });
})();

function openSettingsAfterLoad() {
  if (!mainWindow) {
    return;
  }
  mainWindow.webContents.once('did-finish-load', () => {
    if (mainWindow) {
      mainWindow.webContents.send('open-settings');
    }
  });
}

function sendDirectoryStructure(monitoredDirectories) {
  if (!mainWindow) {
    return;
  }
  const directoryStructure = videoUtils.buildDirectoryStructure(monitoredDirectories);
  mainWindow.webContents.send('directory-structure-updated', directoryStructure);
}

function summarizeMonitoredDirectories(monitoredDirectories = []) {
  const summary = {
    directory_count: Array.isArray(monitoredDirectories) ? monitoredDirectories.length : 0,
    network_path_count: 0,
    has_custom_labels: 0
  };
  if (!Array.isArray(monitoredDirectories)) {
    return summary;
  }
  monitoredDirectories.forEach((entry) => {
    const dirPath = typeof entry === 'string' ? entry : entry && entry.path;
    if (typeof dirPath === 'string') {
      const normalized = dirPath.replace(/\\/g, '/');
      if (normalized.startsWith('\\\\') || normalized.startsWith('//')) {
        summary.network_path_count += 1;
      }
    }
    if (entry && typeof entry === 'object' && entry.name && entry.path && entry.name !== path.basename(entry.path)) {
      summary.has_custom_labels += 1;
    }
  });
  return summary;
}

function trackMainEvent(eventName, properties = {}, options = {}) {
  if (!eventName) {
    return;
  }
  analytics.captureMainEvent(eventName, properties, options);
}

function getTranscoderPort() {
  const envPort = Number(process.env.TRANSCODER_PORT);
  if (Number.isFinite(envPort) && envPort > 0) {
    return envPort;
  }
  return 4455;
}

function bootstrapDirectories() {
  const settings = settingsStore.loadSettings();
  if (!settings.monitoredDirectories || settings.monitoredDirectories.length === 0) {
    openSettingsAfterLoad();
    return;
  }

  directoryWatcher.startWatchingDirectories(settings.monitoredDirectories);
  lastDirectorySummary = summarizeMonitoredDirectories(settings.monitoredDirectories);
  trackMainEvent('directory_watch_started', lastDirectorySummary);
  sendDirectoryStructure(settings.monitoredDirectories);
}

function createWindow() {
  mainWindow = new BrowserWindow({
    width: 1400,
    height: 900,
    title: 'MiBa Video Manager',
    icon: iconPath,
    webPreferences: {
      nodeIntegration: true,
      contextIsolation: false
    }
  });

  mainWindow.loadFile('index.html');
  configureYouTubeRequestHeaders();
  createMenu(mainWindow);

  // Open DevTools in development mode or when DEBUG env var is set
  if (process.env.NODE_ENV === 'development' || process.env.DEBUG === '1' || !app.isPackaged) {
    mainWindow.webContents.openDevTools();
  }

  const registrySnapshot = analytics.getRegistrySnapshot();
  if (registrySnapshot && mainWindow && mainWindow.webContents) {
    mainWindow.webContents.once('did-finish-load', () => {
      if (mainWindow && mainWindow.webContents && !mainWindow.webContents.isDestroyed()) {
        mainWindow.webContents.send('analytics.registry.updated', registrySnapshot);
      }
    });
  }

  bootstrapDirectories();
}

function configureYouTubeRequestHeaders() {
  const targetSession = session.defaultSession;
  if (!targetSession || targetSession._youtubeHeadersConfigured) {
    return;
  }
  targetSession._youtubeHeadersConfigured = true;

  const addHeaders = (details, callback, mutateHeaders) => {
    const requestUrl = details.url || '';
    if (!/https:\/\/(\w+\.)?(youtube\.com|youtube-nocookie\.com)\//i.test(requestUrl)) {
      callback({ cancel: false, requestHeaders: details.requestHeaders });
      return;
    }
    const headers = { ...details.requestHeaders };
    mutateHeaders(headers);
    callback({ cancel: false, requestHeaders: headers });
  };

  targetSession.webRequest.onBeforeSendHeaders((details, callback) => {
    addHeaders(details, callback, (headers) => {
      const refererUrl = headers.Referer || headers.referer;
      if (!refererUrl) {
        const appOrigin = 'https://miba-video-manager.local';
        headers.Referer = appOrigin;
        headers.referer = appOrigin;
      }
      headers['Referrer-Policy'] = 'strict-origin-when-cross-origin';
    });
  });
}

unsubscribeRegistryListener = analytics.onRegistryUpdated((snapshot) => {
  const targetWindow = mainWindow && !mainWindow.isDestroyed() ? mainWindow : null;
  if (targetWindow && targetWindow.webContents && !targetWindow.webContents.isDestroyed()) {
    targetWindow.webContents.send('analytics.registry.updated', snapshot);
  }
});

app.whenReady().then(async () => {
  if (transcoderModule && typeof transcoderModule.startTranscoder === 'function') {
    try {
      await transcoderModule.startTranscoder();
      trackMainEvent('transcoder_started', { port: getTranscoderPort() });
      if (typeof transcoderModule.stopTranscoder === 'function') {
        stopTranscoderService = async () => {
          try {
            await transcoderModule.stopTranscoder();
            trackMainEvent('transcoder_stopped', { port: getTranscoderPort() });
          } catch (error) {
            logger.warn('Failed to stop live transcoder service:', error && error.message ? error.message : error);
            trackMainEvent('transcoder_stop_failed', {
              error_message: error && error.message ? error.message : String(error)
            });
          }
        };
      }
    } catch (error) {
      logger.warn('Failed to auto-start live transcoder service:', error && error.message ? error.message : error);
      trackMainEvent('transcoder_start_failed', {
        error_message: error && error.message ? error.message : String(error)
      });
    }
  }

  // YouTube records will be loaded by the renderer process on startup
  // No need to preload here since we don't have accountId at this point

  createWindow();

  if (process.platform === 'win32') {
    updateWindowsShortcutIcons(iconPath);
  }

  app.on('activate', () => {
    if (BrowserWindow.getAllWindows().length === 0) {
      createWindow();
    }
  });
});

app.on('window-all-closed', () => {
  if (process.platform !== 'darwin') {
    app.quit();
  }
});

app.on('will-quit', () => {
  if (typeof stopTranscoderService === 'function') {
    trackMainEvent('transcoder_stop_requested', { port: getTranscoderPort() });
    stopTranscoderService();
  }
  if (directoryWatcher && typeof directoryWatcher.stopAllWatchers === 'function') {
    trackMainEvent('directory_watch_stopped', lastDirectorySummary);
    directoryWatcher.stopAllWatchers();
  }
  if (typeof unsubscribeRegistryListener === 'function') {
    unsubscribeRegistryListener();
    unsubscribeRegistryListener = null;
  }
  trackMainEvent('app_will_quit', { platform: process.platform });
  analytics.flushQueuedEvents()
    .catch(() => {})
    .finally(() => {
      analytics.shutdownAnalytics();
    });
});

